<?php

namespace FASTRANSPLUGIN\Element;

use Elementor\Controls_Manager;
use Elementor\Controls_Stack;
use Elementor\Group_Control_Typography;
use Elementor\Scheme_Typography;
use Elementor\Scheme_Color;
use Elementor\Group_Control_Border;
use Elementor\Repeater;
use Elementor\Widget_Base;
use Elementor\Utils;
use Elementor\Group_Control_Text_Shadow;
use Elementor\Plugin;

/**
 * Elementor button widget.
 * Elementor widget that displays a button with the ability to control every
 * aspect of the button design.
 *
 * @since 1.0.0
 */
class Cta_Section extends Widget_Base {

	/**
	 * Get widget name.
	 * Retrieve button widget name.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'fastrans_cta';
	}

	/**
	 * Get widget title.
	 * Retrieve button widget title.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'CTA', 'fastrans' );
	}

	/**
	 * Get widget icon.
	 * Retrieve button widget icon.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-library-open';
	}

	/**
	 * Get widget categories.
	 * Retrieve the list of categories the button widget belongs to.
	 * Used to determine where to display the widget in the editor.
	 *
	 * @since  2.0.0
	 * @access public
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'fastrans' ];
	}
	
	/**
	 * Register button widget controls.
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since  1.0.0
	 * @access protected
	 */
	protected function _register_controls() {
		$this->start_controls_section(
			'cta_section',
			[
				'label' => esc_html__( 'CTA Section', 'fastrans' ),
			]
		);
		$repeater = new Repeater();
		
        $repeater->add_group_control(
			\Elementor\Group_Control_Background::get_type(),
			[
				'name' => 'background',
				'label' => __( 'Background', 'fastrans' ),
				'types' => [ 'classic', 'gradient', 'video' ],
				'selector' => '{{WRAPPER}} {{CURRENT_ITEM}}',
			]
		);
		$repeater->add_control(
			'title', [
				'label'       => esc_html__( 'Title', 'fastrans' ),
				'type'        => Controls_Manager::TEXT,
				'label_block' => true,
			]
		);
		$repeater->add_control(
			'description', [
				'label'       => esc_html__( 'Description', 'fastrans' ),
				'type'        => Controls_Manager::TEXTAREA,
				'label_block' => true,
			]
		);
		$repeater->add_control(
			'btn_text', [
				'label'       => esc_html__( 'Button Link', 'fastrans' ),
				'type'        => Controls_Manager::TEXT,
				'label_block' => true,
			]
		);
		$repeater->add_control(
			'btn_link',
			[
				'label' => __( 'Button Link', 'fastrans' ),
				'type' => Controls_Manager::URL,
				'placeholder' => __( 'https://your-link.com', 'fastrans' ),
			]
		);
		$this->add_control(
			'cta_items',
			[
				'label'       => __( 'Add Item', 'fastrans-extension' ),
				'type'        => Controls_Manager::REPEATER,
				'fields'      => $repeater->get_controls(),
				'title_field' => '{{{ title }}}',
			]
		);

		$this->end_controls_section();
		$this->start_controls_section(
			'cta_style_main',
			[
				'label' => esc_html__( 'Cta Style', 'fastrans' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);
		$this->add_control(
			'cta_style',
			[
				'type'      => Controls_Manager::HEADING,
				'label'     => esc_html__( 'CTA Style', 'fastrans' ),
				'separator' => 'before',
			]
		);
		$this->add_control(
			'ct_title_clr',
			[
				'label'     => esc_html__( 'Title Color', 'fastrans' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .fs-cta-item-box h1' => 'color: {{VALUE}}',
				],
			]
		);
		
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'           => 'ct_title_typography',
				'label'          => esc_html__( 'Typography', 'fastrans' ),
				'selector'       => '{{WRAPPER}} .fs-cta-item-box h1',
				'fields_options' => [
					'typography' => [
						'default' => 'custom',
					],
				],
			]
		);
        $this->add_control(
			'cta_text',
			[
				'type'      => Controls_Manager::HEADING,
				'label'     => esc_html__( 'CTA Text Style', 'fastrans' ),
				'separator' => 'before',
			]
		);
		$this->add_control(
			'ct_text_clr',
			[
				'label'     => esc_html__( 'Text Color', 'fastrans' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .fs-cta-item-box p' => 'color: {{VALUE}}',
				],
			]
		);
		
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'           => 'ct_text_typography',
				'label'          => esc_html__( 'Typography', 'fastrans' ),
				'selector'       => '{{WRAPPER}} .fs-cta-item-box p',
				'fields_options' => [
					'typography' => [
						'default' => 'custom',
					],
				],
			]
		);
        $this->add_control(
			'cta_btn_text',
			[
				'type'      => Controls_Manager::HEADING,
				'label'     => esc_html__( 'CTA Btn Style', 'fastrans' ),
				'separator' => 'before',
			]
		);
		$this->add_control(
			'ct_btn_clr',
			[
				'label'     => esc_html__( 'Button Color', 'fastrans' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .fs-cta-item-box a' => 'color: {{VALUE}}',
				],
			]
		);
		$this->add_control(
			'ct_btn_bg_clr',
			[
				'label'     => esc_html__( 'Button BG Color', 'fastrans' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .fs-cta-item-box a' => 'background-color: {{VALUE}}',
				],
			]
		);
		
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'           => 'btn_typography',
				'label'          => esc_html__( 'Typography', 'fastrans' ),
				'selector'       => '{{WRAPPER}} .fs-cta-item-box a',
				'fields_options' => [
					'typography' => [
						'default' => 'custom',
					],
				],
			]
		);
		
		$this->end_controls_section();
	}

	/**
	 * Render button widget output on the frontend.
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since  1.0.0
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();
		$cta_items = $settings['cta_items'];
		?>
		
        <!-- Skill Section
        ============================================= -->
        <section id="ft-cta-section" class="ft-cta-section">
			<div class="container-fluid">
				<div class="row">
                    <?php foreach($cta_items as $item):?>
					<div class="col-lg-6">
                        <div class="fs-cta-item-box text-center elementor-repeater-item-<?php echo esc_attr($item['_id']);?>">
                            <h1><?php echo esc_html($item['title']);?></h1>
                            <p><?php echo __($item['description']);?></p>
                            <a href="<?php echo esc_url($item['btn_link']['url']);?>"><?php echo esc_html($item['btn_text']);?></a>
                        </div>
                    </div>
                    <?php endforeach;?>
				</div>
			</div>
        </section>	
    	<!-- Skill Section
        ============================================= -->
                         
        <?php
	}

}
